

Type TPhysicsSimulator
	
'#Region Private members
	Field _gravity:Vector2
	
	Field _arbiterPool:TArbiterPool
	Field _arbiterPoolSize:Int = 10
	
	Field _contactPool:TContactPool
	Field _contactPoolSize:Int = 100
	
	Field _geomList:TGeomList
	Field _geomAddList:TList
	Field _geomRemoveList:TList
	
	Field _bodyList:TBodyList
	Field _bodyAddList:TList
	Field _bodyRemoveList:TList
	
	Field _controllerList:TControllerList
	Field _controllerAddList:TList
	Field _controllerRemoveList:TList
	
	Field _jointList:TJointList
	Field _jointAddList:TList
	Field _jointRemoveList:TList
	
	Field _arbiterList:TArbiterList
	
	' default settings
	Field _iterations:Int = 5
	Field _allowedPenetration:Float =.05
	Field _biasFactor:Float =.8
	Field _maxContactsToDetect:Int = 10
	Field _maxContactsToResolve:Int = 4
	Field _frictionType:Int = FrictionType.Average
	
	'performance diagnostics
	Field UpdateTime:Int = -1
	Field CleanUptime:Int = -1
	Field BroadPhaseCollisionTime:Int = -1
	Field NarrowPhaseCollisionTime:Int = -1
	Field ApplyForcesTime:Int = -1
	Field ApplyImpulsesTime:Int = -1
	Field UpdatePositionsTime:Int = -1
	Field EnableDiagnostics:Int = False
	
	Field _broadPhaseCollider:IBroadPhaseCollider
'#End Region 
	
'#Region Public Properties (getter/setters)
	rem
	bbdoc: should be treated as readonly. do not directly add/remove from this list.
	end rem
	Method GetGeomList:TGeomList() 
		Return _geomList
	End Method
	
	rem
	bbdoc: should be treated as readonly. do not directly add/remove from this list.
	end rem
	Method GetBodyList:TBodyList() 
		Return _bodyList
	End Method
	
	rem
	bbdoc: should be treated as readonly. do not directly add/remove from this list.
	end rem
	Method GetControllerList:TControllerList() 
		Return _controllerList
	End Method
	
	rem
	bbdoc: should be treated as readonly. do not directly add/remove from this list.
	end rem
	Method GetJointList:TJointList() 
		Return _jointList
	End Method
	
	rem
	bbdoc: should be treated as readonly. do not directly add/remove from this list.
	end rem
	Method GetArbiterList:TArbiterList() 
		Return _arbiterList
	End Method
	
	
	Method GetGravity:Vector2() 
		Return _gravity.Copy() 
	End Method
	
	Method SetGravity(value:Vector2) 
		_gravity.Set(value) 
	End Method
	
	Method GetIterations:Int() 
		Return _iterations
	End Method
	
	Method SetIterations(value:Int) 
		_iterations = value
	End Method
	
	Method GetAllowedPenetration:Float() 
		Return _allowedPenetration
	End Method
	
	Method SetAllowedPenetration(value:Float) 
		 _allowedPenetration = value
	End Method
	
	Method GetBiasFactor:Float() 
		Return _biasFactor
	End Method
	
	Method SetBiasFactor(value:Float) 
		_biasFactor = value
	End Method
	
	Method GetMaxContactsToDetect:Int() 
		Return _maxContactsToDetect
	End Method
	
	Method SetMaxContactsToDetect(value:Int) 
		_maxContactsToDetect = value
	End Method
	
	Method SetMaxContactsToResolve(value:Int)
		_maxContactsToResolve = value
	End Method
	
	Method GetMaxContactsToResolve:Int()
		Return _maxContactsToResolve
	End Method
	
	Method GetFrictionType:Int() 
		Return _frictionType
	End Method
	
	rem
	bbdoc: use FrictionType enum class.
	end rem
	Method SetFrictionType(value:Int) 
		_frictionType = value
	End Method
'#End Region 
	
'#Region Construction/init
	Method New() 
		_gravity = Vector2.Create(0, 0) 
	End Method
	
	Function Create:TPhysicsSimulator(gravity:Vector2) 
		Local ps:TPhysicsSimulator = New TPhysicsSimulator
		ps._gravity = gravity.Copy() 
		ps._Init() 
		Return ps
	End Function
	
	Method _Init() 
		_geomList = New TGeomList
		_geomAddList = CreateList() 
		_geomRemoveList = CreateList() 
		
		_bodyList = New TBodyList
		_bodyAddList = CreateList() 
		_bodyRemoveList = CreateList() 
		
		_controllerList = New TControllerList
		_controllerAddList = CreateList() 
		_controllerRemoveList = CreateList() 
		
		_jointList = New TJointList
		_jointAddList = CreateList() 
		_jointRemoveList = CreateList() 
		
		_arbiterList = New TArbiterList	
		_arbiterPool = TArbiterPool.Create(_arbiterPoolSize) 
		
		_contactPool = TContactPool.Create(_contactPoolSize) 
		_broadPhaseCollider = TSelectiveSweepCollider.Create(Self) 
	End Method
'#End Region 

'#Region Public Methods

	Method SetBroadPhaseCollider(broadPhaseCollider:IBroadPhaseCollider) 
		If _geomList.Count() > 0 Then
			Throw "The Geomlist must be empty when setting the broad phase collider type"
		End If
		Self._broadPhaseCollider = broadPhaseCollider
	End Method
	
	'#Region Add/Remove Methods
	Method AddGeom(geometry:TGeom) 
		If Not _geomAddlist.Contains(geometry) Then
			_geomAddList.AddLast(geometry) 
		End If
	End Method
	
	Method RemoveGeom(geometry:TGeom) 
		_geomRemoveList.AddLast(geometry) 
	End Method
	
	Method AddBody(body:TBody) 
		If Not _bodyAddList.Contains(body) Then
			_bodyAddList.AddLast(body) 
		End If
	End Method
	
	Method RemoveBody(body:TBody) 
		_bodyRemoveList.AddLast(body) 
	End Method
	
	Method AddController(controller:TController) 
		If Not _controllerAddList.Contains(controller) Then
			_controllerAddList.AddLast(controller) 
		End If
	End Method
	
	Method RemoveController(controller:TController) 
		_controllerRemoveList.AddLast(controller) 
	End Method
	
	Method AddJoint(joint:TJoint) 
		If Not _jointAddList.Contains(joint) Then
			_jointAddList.AddLast(joint) 
		End If
	End Method
	
	Method RemoveJoint(joint:TJoint) 
		_jointRemoveList.AddLast(joint) 
	End Method
	
	'TODO: wtf why doesn't a string tag equate to a string value object w/o a cast/tostring()?!
	Method GetBody:TBody(value:Object) 
		For Local body:TBody = EachIn _bodyList
			If body._tag <> Null Then
				If body._tag.ToString() = value.ToString() Then
					Return body
				End If
			End If
		Next
		Return Null
	End Method
	
	Method Clear() 
		_arbiterList.Clear() 
		_geomList.Clear() 
		_bodylist.Clear() 
		_jointlist.Clear() 
		_controllerList.Clear() 
	End Method
	'#End Region 
	
	Method Update(dt:Float) 
		
		If dt = 0 Then Return
		Local ms:Int = MilliSecs() 
		_ProcessAddedItems() 
		_ProcessRemovedItems() 
		_ProcessDisposedItems() 
		If EnableDiagnostics Then CleanUpTime = MilliSecs() - ms
		_DoBroadPhaseCollision()
		If EnableDiagnostics Then BroadPhaseCollisionTime = MilliSecs() - ms - CleanUpTime
		_DoNarrowPhaseCollision()		
		If EnableDiagnostics Then NarrowPhaseCollisionTime = MilliSecs() - ms - BroadPhaseCollisionTime - CleanUpTime
		_ApplyForces(dt)
		If EnableDiagnostics Then ApplyForcesTime = MilliSecs() - ms - NarrowPhaseCollisionTime - BroadPhaseCollisionTime - CleanUpTime		
		_ApplyImpulses(dt)
		If EnableDiagnostics Then ApplyImpulsesTime = MilliSecs() - ms - ApplyForcesTime - NarrowPhaseCollisionTime - BroadPhaseCollisionTime - CleanUpTime
		_UpdatePositions(dt)
		If EnableDiagnostics Then UpdatePositionsTime = MilliSecs() - ms - ApplyImpulsesTime - ApplyForcesTime - NarrowPhaseCollisionTime - BroadPhaseCollisionTime - CleanUpTime
		
		If EnableDiagnostics Then
			UpdateTime = MilliSecs() - ms			
		End If
	End Method
	
	rem
	bbdoc: given a coordinate this will return
	a geometry object if it intersects with this point
	end rem
	Method Collide:TGeom(point:Vector2)
		For Local geom:TGeom = EachIn _geomList
			If geom.Collide(point) Then
				Return geom
			End If
		Next
		Return Null
	End Method
'#End Region 

'#Region Private Methods
	
	Method _DoBroadPhaseCollision() 
		
		
		Self._broadPhaseCollider.Update() 
		
		
	End Method
	
	Method _DoNarrowPhaseCollision() 
	
		For Local arbiter:TArbiter = EachIn _arbiterlist
			arbiter.Collide() 
		Next
		
		_arbiterList.RemoveContactCountEqualsZero(Self._arbiterPool) 
		
	End Method
	
	Field _gravityForce:Vector2 = Vector2.Zero() 
	Method _ApplyForces(dt:Float) 
		For Local controller:TController = EachIn _controllerList
			If controller.IsEnabled() Then
				controller.Update(dt)
			End If
		Next
		
		For Local body:TBody = EachIn _bodyList
			If Not body._enabled Then
				Continue
			End If
			body.ApplyImpulses() 
			
			If Not body._ignoreGravity Then
				_gravityForce.X = _gravity.X * body._mass
				_gravityForce.Y = _gravity.Y * body._mass
				
				'#region INLINE:body.ApplyForce(ref gravityForce) ;
				body._force.X = body._force.X + _gravityForce.X;
				body._force.Y = body._force.Y + _gravityForce.Y;
				'#endregion
			End If
			body.IntegrateVelocity(dt) 
			body.ClearForce() 
			body.ClearTorque() 
		Next
	End Method
	
	Method _ApplyImpulses(dt:Float) 
		Local inverseDt:Float = 1.0 / dt
		For Local joint:TJoint = EachIn _jointList
			joint.PreStep(inverseDt)
		Next
		
 
		For Local arbiter:TArbiter = EachIn _arbiterList
			arbiter.PreStepImpulse(inverseDt) 
		Next
		

		For Local h:Int = 1 To _iterations
			For Local joint:TJoint = EachIn _jointList
				joint.Update()
			Next
			
			For Local arbiter:TArbiter = EachIn _arbiterList			
				arbiter.ApplyImpulse() 
			Next
		Next
	End Method
	
	
	Method _UpdatePositions(dt:Float)
		
		For Local body:TBody = EachIn _bodyList
			If Not(body._enabled) Then Continue
			body.IntegratePosition(dt) 
		Next
	End Method
	
	Method _ReleaseArbitersWithContactCountZero(arbiter:TArbiter) 
		If arbiter.GetContactCount() = 0 Then
			Self._arbiterPool.Free(arbiter)
		End If
	End Method
	'#Region process removed/added items
	Method _ProcessAddedItems() 
		' add geometry 
		For Local geom:TGeom = EachIn _geomAddList
			If Not _geomList.Contains(geom) Then
				geom.isRemoved = False				
				_geomList.add(geom) 
				_broadPhaseCollider.Add(geom)
			End If
		Next
		_geomAddList.Clear() 
		
		' add bodies
		For Local body:TBody = EachIn _bodyAddList
			If Not _bodylist.Contains(body) Then _bodyList.add(body) 
		Next
		_bodyAddList.Clear() 
		
		' add controllers
		For Local controller:TController = EachIn _controllerAddList
			If Not _controllerList.Contains(controller) Then _controllerList.add(controller) 
		Next		
		_controllerAddList.Clear() 
		
		' add joints
		For Local joint:TJoint = EachIn _jointAddList
			If Not _jointList.Contains(joint) Then _jointlist.add(joint) 
		Next
		_jointAddList.Clear() 
	End Method
	
	Method _ProcessRemovedItems() 
		' remove any geometries
		For Local geom:TGeom = EachIn _geomRemoveList
			geom.isRemoved = True
			_geomlist.Remove(geom) 
			geom.ClearListeners() ' remove event handlers
			
			' remove any arbiters associated with this geometry
			For Local arbiter:TArbiter = EachIn _arbiterlist
				If arbiter.geometryA = geom Or arbiter.geometryB = geom Then
					_arbiterList.Remove(arbiter) 
				End If
			Next
		Next
		
		If _geomRemoveList.Count() > 0 Then
			Self._broadPhaseCollider.ProcessRemovedGeoms() 
		End If
		
		_geomRemovelist.Clear() 
		
		' remove bodies
		For Local body:TBody = EachIn _bodyRemoveList
			_bodyList.Remove(body)
			body.updateHandler.Clear()	'remove event handlers
		Next
		_bodyRemoveList.Clear() 
		
		' remove controllers
		For Local controller:TController = EachIn _controllerRemoveList
			_controllerList.Remove(controller) 
		Next
		_controllerRemoveList.Clear() 
		
		' remove joints
		For Local joint:TJoint = EachIn _jointRemoveList
			_jointList.Remove(joint) 
		Next
		_jointRemovelist.Clear() 
		
	End Method
	
	rem
	bbdoc: I don't think this method is necessary in the Bmax world.
	end rem
	Method _ProcessDisposedItems() 
'		Local disposedGeomCount:Int
'		' allow each controller to validate itself. this is where a controller can dispose of itself if need be.
'		For Local controller:TController = EachIn _controllerlist
'			controller.Validate() 
'		Next
'		disposedGeomCount = _geomList.RemoveDisposed() 
'		
'		If disposedGeomCount > 0 Then Self._broadPhaseCollider.ProcessDisposedGeoms() 
'		
'		_bodylist.RemoveDisposed() 
'		_controllerList.RemoveDisposed() 
	End Method
	'#End Region 
'#End Region 
End Type
